/*-------------------------------------------------------------*/
/* NOM........  sha.c                                          */
/* OBJET......  Secure Hash Algorithm and K-HMAC               */
/* PROJET.....  SFP V.1                                        */
/* REFERENCES.                                                 */
/* VERSION....  1.3                                            */
/* HISTORIQUE.  BODJ - 09.05.2005 - modification Is64or32(...) */
/*              afin de dtecter correctement l'architecture   */
/*              du serveur (32/64 bits)                        */
/*-------------------------------------------------------------*/

/*---------------------------------------------------------*/
/*     Include Files                                       */
/*---------------------------------------------------------*/
#include <string.h>
#include <stdio.h>
#include "limits.h"
#include "float.h"
#include "sha.h"

#if !defined(MAXSHORT)
#define MAXSHORT SHRT_MAX
#endif

#if !defined(MAXLONG)
#define MAXLONG LONG_MAX
#endif

/*---------------------------------------------------------*/
/*     Prototypes definition                               */
/*---------------------------------------------------------*/
int Is64or32(void);
void ShaProc64(unsigned char*,int,unsigned char*);
void ShaProc32(unsigned char*,int,unsigned char*);


/*---------------------------------------------------------*/
/*     Global variables                                    */
/*---------------------------------------------------------*/
/* Declaration for 64 bits patform */
static unsigned int W64[80];               /* Word Input Sequence                    */
static unsigned int H64[5];                /* 160 Bits Message Digest                */
static unsigned int A64,B64,C64,D64,E64;   /* A, B, C, D, E Registers for 80 Rounds  */

/* Declaration for 32 bits patform */
static unsigned long W32[80];              /* Word Input Sequence                   */
static unsigned long H32[5];               /* 160 Bits Message Digest               */
static unsigned long A32,B32,C32,D32,E32;  /* A, B, C, D, E Registers for 80 Rounds */

static unsigned char lastBlock[64];        /* Last Block To Hash                    */


/*---------------------------------------------------------*/
/*     Is64or32()                                          */
/*---------------------------------------------------------*/
int Is64or32()
{
  if (MAXLONG>0x7FFF) return 64;else return 32;
}


/*---------------------------------------------------------*/
/*     ShaProc()                                           */
/*     Secure Hash Algorithm Computing (Length < 32K)      */
/*---------------------------------------------------------*/
void ShaProc(unsigned char* entry,int len,unsigned char* exit)
{
  if (Is64or32()==64) ShaProc64(entry,len,exit);
    else ShaProc32(entry,len,exit);
}


/*---------------------------------------------------------*/
/*     ShaProc64()                                         */
/*     Secure Hash Algorithm Computing (Length < 32K)      */
/*---------------------------------------------------------*/
void ShaProc64(unsigned char* entry,int len,unsigned char* exit)
{
  unsigned int iter;  /* Number of 512 Bits Blocks */
  int lg;             /* Length To Process         */
  unsigned int val;   /* Temporary Value           */
  unsigned char* pe;  /* Entry Pointer             */
  int i;              /* Item Loop                 */

  /* 1. Initialization */
  H64[0]=0x67452301L;
  H64[1]=0xEFCDAB89L;
  H64[2]=0x98BADCFEL;
  H64[3]=0x10325476L;
  H64[4]=0xC3D2E1F0L;
  iter=(len+72)>>6;

  /* 2. Message Block Processing */
  for(pe=entry,lg=len; iter>0; iter--,lg-=64){

    /* 2.1 Message Padding */
    if (lg<64)
    {
      memset(lastBlock,0,sizeof(lastBlock));
      if (lg>0)   memcpy(lastBlock,pe,lg);
      if (lg>=0)  lastBlock[lg]=0x80;
      if (lg<=55)
      {
        lastBlock[61]=(unsigned char)(len>>13);
        lastBlock[62]=(unsigned char)(len>>5);
        lastBlock[63]=(unsigned char)((len<<3)&0xFF);
      }
      pe=lastBlock;
    }

    /* 2.2 Divide Mi Into W0,...,W15 (step a) */
    for(i=0; i<64; i++)
      W64[i>>2]=(W64[i>>2]<<8)+*pe++;
    /* 2.3 Processing W16 To W79 - Init A To E (step b c) */
    for(i=16; i<80; i++)
    {
      val=W64[i-3]^W64[i-8]^W64[i-14]^W64[i-16];
      W64[i]=(val<<1)+(val>>31);
    }
    A64=H64[0]; B64=H64[1]; C64=H64[2]; D64=H64[3]; E64=H64[4];

    /* 2.4 Round 0 To 79 (step d) */
    for(i=0; i<20; i++)
    {
      val=(A64<<5)+(A64>>27)+((B64&C64)|((~B64)&D64))+E64+W64[i]+0x5A827999L;
      E64=D64; D64=C64; C64=(B64<<30)+(B64>>2); B64=A64; A64=val;
    }
    for(; i<40; i++)
    {
      val=(A64<<5)+(A64>>27)+(B64^C64^D64)+E64+W64[i]+0x6ED9EBA1L;
      E64=D64; D64=C64; C64=(B64<<30)+(B64>>2); B64=A64; A64=val;
    }
    for(; i<60; i++)
    {
      val=(A64<<5)+(A64>>27)+((B64&C64)|(B64&D64)|(C64&D64))+E64+W64[i]+0x8F1BBCDCL;
      E64=D64; D64=C64; C64=(B64<<30)+(B64>>2); B64=A64; A64=val;
    }
    for(; i<80; i++)
    {
      val=(A64<<5)+(A64>>27)+(B64^C64^D64)+E64+W64[i]+0xCA62C1D6L;
      E64=D64; D64=C64; C64=(B64<<30)+(B64>>2); B64=A64; A64=val;
    }

    /* 2.5 Update Digest (step e) */
    H64[0]+=A64; H64[1]+=B64; H64[2]+=C64; H64[3]+=D64; H64[4]+=E64;
  }

  /* 3. Message Digest */
  for(i=19,exit+=19; i>=0; i--)
  {
    *(exit--)=(unsigned char)(H64[i>>2]&0xFF);
    H64[i>>2]>>=8;
  }
}


/*---------------------------------------------------------*/
/*     ShaProc32()                                         */
/*     Secure Hash Algorithm Computing (Length < 32K)      */
/*---------------------------------------------------------*/
void ShaProc32(unsigned char* entry,int len,unsigned char* exit)
{
  unsigned int iter;  /* Number of 512 Bits Blocks */
  int lg;             /* Length To Process         */
  unsigned long val;  /* Temporary Value           */
  unsigned char* pe;  /* Entry Pointer             */
  int i;              /* Item Loop                 */

  /* 1. Initialization */
  H32[0]=0x67452301L;
  H32[1]=0xEFCDAB89L;
  H32[2]=0x98BADCFEL;
  H32[3]=0x10325476L;
  H32[4]=0xC3D2E1F0L;
  iter=(len+72)>>6;

  /* 2. Message Block Processing */
  for(pe=entry,lg=len; iter>0; iter--,lg-=64)
  {

    /* 2.1 Message Padding */
    if (lg<64)
    {
      memset(lastBlock,0,sizeof(lastBlock));
      if (lg>0)   memcpy(lastBlock,pe,lg);
      if (lg>=0)  lastBlock[lg]=0x80;
      if (lg<=55)
      {
        lastBlock[61]=(unsigned char)(len>>13);
        lastBlock[62]=(unsigned char)(len>>5);
        lastBlock[63]=(unsigned char)((len<<3)&0xFF);
      }
      pe=lastBlock;
    }

    /* 2.2 Divide Mi Into W0,...,W15 (step a) */
    for(i=0; i<64; i++)
      W32[i>>2]=(W32[i>>2]<<8)+*pe++;
    /* 2.3 Processing W16 To W79 - Init A To E (step b c) */
    for(i=16; i<80; i++)
    {
      val=W32[i-3]^W32[i-8]^W32[i-14]^W32[i-16];
      W32[i]=(val<<1)+(val>>31);
    }
    A32=H32[0]; B32=H32[1]; C32=H32[2]; D32=H32[3]; E32=H32[4];

    /* 2.4 Round 0 To 79 (step d) */
    for(i=0; i<20; i++)
    {
      val=(A32<<5)+(A32>>27)+((B32&C32)|((~B32)&D32))+E32+W32[i]+0x5A827999L;
      E32=D32; D32=C32; C32=(B32<<30)+(B32>>2); B32=A32; A32=val;
    }
    for(; i<40; i++)
    {
      val=(A32<<5)+(A32>>27)+(B32^C32^D32)+E32+W32[i]+0x6ED9EBA1L;
      E32=D32; D32=C32; C32=(B32<<30)+(B32>>2); B32=A32; A32=val;
    }
    for(; i<60; i++)
    {
      val=(A32<<5)+(A32>>27)+((B32&C32)|(B32&D32)|(C32&D32))+E32+W32[i]+0x8F1BBCDCL;
      E32=D32; D32=C32; C32=(B32<<30)+(B32>>2); B32=A32; A32=val;
    }
    for(; i<80; i++)
    {
      val=(A32<<5)+(A32>>27)+(B32^C32^D32)+E32+W32[i]+0xCA62C1D6L;
      E32=D32; D32=C32; C32=(B32<<30)+(B32>>2); B32=A32; A32=val;
    }

    /* 2.5 Update Digest (step e) */
    H32[0]+=A32; H32[1]+=B32; H32[2]+=C32; H32[3]+=D32; H32[4]+=E32;
  }

  /* 3. Message Digest */
  for(i=19,exit+=19; i>=0; i--)
  {
    *(exit--)=(unsigned char)(H32[i>>2]&0xFF);
    H32[i>>2]>>=8;
  }
}


/*---------------------------------------------------------*/
/*     Hmac()                                              */
/*     SHA-1 Keyed Hash MAC                                */
/*---------------------------------------------------------*/
void Hmac(unsigned char* key,int keyLn,unsigned char* text,int textLn,unsigned char* out)
{
  unsigned char *pad;  /* Padding      */
  int i;               /* Item Loop    */
  unsigned char* pc;   /* Data Pointer */
  
  pad = (unsigned char *)malloc((textLn+64+1)*sizeof(unsigned char));
  memset(pad,0x36,64);
  for(i=0,pc=key; i<keyLn; )
    pad[i++]^=*pc++;
  memcpy(pad+64,text,textLn);
  ShaProc(pad,64+textLn,pad+64);
  memset(pad,0x5C,64);
  for(i=0,pc=key; i<keyLn; )
    pad[i++]^=*pc++;
  ShaProc(pad,64+20,out);
  free(pad);
}


/*---------------------------------------------------------*/
/*     ConvertInAsciiStream()                              */
/*---------------------------------------------------------*/
void ConvertInAsciiStream (unsigned char* bufIn, unsigned char bufLg, unsigned char *out)
{
  int i;  /* Item Loop */
  unsigned char v;

  for(i=0; i<bufLg; i++)
  {
    v=(unsigned char)(*bufIn>>4);
    if (v>9) *out++=(unsigned char)(v+0x37); else *out++=(unsigned char)(v+0x30);
    v=(unsigned char)(*bufIn++&0x0f);
    if (v>9) *out++=(unsigned char)(v+0x37); else *out++=(unsigned char)(v+0x30);
  }
}
