/*-----------------------------------------------------------*/
/* NOM........  des.c                                        */
/* OBJET......  DES Cryptography                             */
/* PROJET.....  SFP V.1                                      */
/* REFERENCES.                                               */
/* VERSION....  1.0                                          */
/* HISTORIQUE.                                               */
/*-----------------------------------------------------------*/

#include <stdio.h>
#include <string.h>


/*-----------------------------------------------------------*/
/*      Definitions de masques 32 bits                       */
/*-----------------------------------------------------------*/

#define BITSPERBYTE 8
#define BITS(type)  (BITSPERBYTE * (int)sizeof (type))
#define HIBITS  ((short)(1 << BITS(short) - 1))
#define MAXSHORT  ((short)~HIBITS)

#define M28 0xFFFFFFF0L
#define B01 0x80000000L
#define B02 0x40000000L
#define B03 0x20000000L
#define B04 0x10000000L
#define B05 0x8000000L
#define B06 0x4000000L
#define B07 0x2000000L
#define B08 0x1000000L
#define B09 0x800000L
#define B10 0x400000L
#define B11 0x200000L
#define B12 0x100000L
#define B13 0x80000L
#define B14 0x40000L
#define B15 0x20000L
#define B16 0x10000L
#define B17 0x8000L
#define B18 0x4000L
#define B19 0x2000L
#define B20 0x1000L
#define B21 0x800L
#define B22 0x400L
#define B23 0x200L
#define B24 0x100L
#define B25 0x80L
#define B26 0x40L
#define B27 0x20L
#define B28 0x10L
#define B29 0x8L
#define B30 0x4L
#define B31 0x2L
#define B32 0x1L


/*-----------------------------------------------------------*/
/*      Declaration de Variables                             */
/*-----------------------------------------------------------*/

static unsigned int      c64[2][16];   /* 16 cles intermediaires de 48 bits, cadrees a gauche  */
static unsigned long     c32[2][16];   /* 16 cles intermediaires de 48 bits, cadrees a gauche  */

static short    s1[64]={    /* 1ere table de substitution de 6 bits en 4 : b4=s[b6] */
        0xE, 0x0, 0x4, 0xF, 0xD, 0x7, 0x1, 0x4,
        0x2, 0xE, 0xF, 0x2, 0xB, 0xD, 0x8, 0x1,
        0x3, 0xA, 0xA, 0x6, 0x6, 0xC, 0xC, 0xB,
        0x5, 0x9, 0x9, 0x5, 0x0, 0x3, 0x7, 0x8,
        0x4, 0xF, 0x1, 0xC, 0xE, 0x8, 0x8, 0x2,
        0xD, 0x4, 0x6, 0x9, 0x2, 0x1, 0xB, 0x7,
        0xF, 0x5, 0xC, 0xB, 0x9, 0x3, 0x7, 0xE,
        0x3, 0xA, 0xA, 0x0, 0x5, 0x6, 0x0, 0xD
           };

static short    s2[64]={    /* 2eme table de substitution de 6 bits en 4 : b4=s[b6] */
        0xF, 0x3, 0x1, 0xD, 0x8, 0x4, 0xE, 0x7,
        0x6, 0xF, 0xB, 0x2, 0x3, 0x8, 0x4, 0xE,
        0x9, 0xC, 0x7, 0x0, 0x2, 0x1, 0xD, 0xA,
        0xC, 0x6, 0x0, 0x9, 0x5, 0xB, 0xA, 0x5,
        0x0, 0xD, 0xE, 0x8, 0x7, 0xA, 0xB, 0x1,
        0xA, 0x3, 0x4, 0xF, 0xD, 0x4, 0x1, 0x2,
        0x5, 0xB, 0x8, 0x6, 0xC, 0x7, 0x6, 0xC,
        0x9, 0x0, 0x3, 0x5, 0x2, 0xE, 0xF, 0x9
           };

static short    s3[64]={    /* 3eme table de substitution de 6 bits en 4 : b4=s[b6] */
        0xA, 0xD, 0x0, 0x7, 0x9, 0x0, 0xE, 0x9,
        0x6, 0x3, 0x3, 0x4, 0xF, 0x6, 0x5, 0xA,
        0x1, 0x2, 0xD, 0x8, 0xC, 0x5, 0x7, 0xE,
        0xB, 0xC, 0x4, 0xB, 0x2, 0xF, 0x8, 0x1,
        0xD, 0x1, 0x6, 0xA, 0x4, 0xD, 0x9, 0x0,
        0x8, 0x6, 0xF, 0x9, 0x3, 0x8, 0x0, 0x7,
        0xB, 0x4, 0x1, 0xF, 0x2, 0xE, 0xC, 0x3,
        0x5, 0xB, 0xA, 0x5, 0xE, 0x2, 0x7, 0xC
           };

static short    s4[64]={    /* 4eme table de substitution de 6 bits en 4 : b4=s[b6] */
        0x7, 0xD, 0xD, 0x8, 0xE, 0xB, 0x3, 0x5,
        0x0, 0x6, 0x6, 0xF, 0x9, 0x0, 0xA, 0x3,
        0x1, 0x4, 0x2, 0x7, 0x8, 0x2, 0x5, 0xC,
        0xB, 0x1, 0xC, 0xA, 0x4, 0xE, 0xF, 0x9,
        0xA, 0x3, 0x6, 0xF, 0x9, 0x0, 0x0, 0x6,
        0xC, 0xA, 0xB, 0x1, 0x7, 0xD, 0xD, 0x8,
        0xF, 0x9, 0x1, 0x4, 0x3, 0x5, 0xE, 0xB,
        0x5, 0xC, 0x2, 0x7, 0x8, 0x2, 0x4, 0xE
           };

static short    s5[64]={    /* 5eme table de substitution de 6 bits en 4 : b4=s[b6] */
        0x2, 0xE, 0xC, 0xB, 0x4, 0x2, 0x1, 0xC,
        0x7, 0x4, 0xA, 0x7, 0xB, 0xD, 0x6, 0x1,
        0x8, 0x5, 0x5, 0x0, 0x3, 0xF, 0xF, 0xA,
        0xD, 0x3, 0x0, 0x9, 0xE, 0x8, 0x9, 0x6,
        0x4, 0xB, 0x2, 0x8, 0x1, 0xC, 0xB, 0x7,
        0xA, 0x1, 0xD, 0xE, 0x7, 0x2, 0x8, 0xD,
        0xF, 0x6, 0x9, 0xF, 0xC, 0x0, 0x5, 0x9,
        0x6, 0xA, 0x3, 0x4, 0x0, 0x5, 0xE, 0x3
           };

static short    s6[64]={    /* 6eme table de substitution de 6 bits en 4 : b4=s[b6] */
        0xC, 0xA, 0x1, 0xF, 0xA, 0x4, 0xF, 0x2,
        0x9, 0x7, 0x2, 0xC, 0x6, 0x9, 0x8, 0x5,
        0x0, 0x6, 0xD, 0x1, 0x3, 0xD, 0x4, 0xE,
        0xE, 0x0, 0x7, 0xB, 0x5, 0x3, 0xB, 0x8,
        0x9, 0x4, 0xE, 0x3, 0xF, 0x2, 0x5, 0xC,
        0x2, 0x9, 0x8, 0x5, 0xC, 0xF, 0x3, 0xA,
        0x7, 0xB, 0x0, 0xE, 0x4, 0x1, 0xA, 0x7,
        0x1, 0x6, 0xD, 0x0, 0xB, 0x8, 0x6, 0xD
           };

static short    s7[64]={    /* 7eme table de substitution de 6 bits en 4 : b4=s[b6] */
        0x4, 0xD, 0xB, 0x0, 0x2, 0xB, 0xE, 0x7,
        0xF, 0x4, 0x0, 0x9, 0x8, 0x1, 0xD, 0xA,
        0x3, 0xE, 0xC, 0x3, 0x9, 0x5, 0x7, 0xC,
        0x5, 0x2, 0xA, 0xF, 0x6, 0x8, 0x1, 0x6,
        0x1, 0x6, 0x4, 0xB, 0xB, 0xD, 0xD, 0x8,
        0xC, 0x1, 0x3, 0x4, 0x7, 0xA, 0xE, 0x7,
        0xA, 0x9, 0xF, 0x5, 0x6, 0x0, 0x8, 0xF,
        0x0, 0xE, 0x5, 0x2, 0x9, 0x3, 0x2, 0xC
           };

static short    s8[64]={    /* 8eme table de substitution de 6 bits en 4 : b4=s[b6] */
        0xD, 0x1, 0x2, 0xF, 0x8, 0xD, 0x4, 0x8,
        0x6, 0xA, 0xF, 0x3, 0xB, 0x7, 0x1, 0x4,
        0xA, 0xC, 0x9, 0x5, 0x3, 0x6, 0xE, 0xB,
        0x5, 0x0, 0x0, 0xE, 0xC, 0x9, 0x7, 0x2,
        0x7, 0x2, 0xB, 0x1, 0x4, 0xE, 0x1, 0x7,
        0x9, 0x4, 0xC, 0xA, 0xE, 0x8, 0x2, 0xD,
        0x0, 0xF, 0x6, 0xC, 0xA, 0x9, 0xD, 0x0,
        0xF, 0x3, 0x3, 0x5, 0x5, 0x6, 0x8, 0xB
           };


static unsigned char vect[]="0x0b,0x0b,0x0b,0x0b,0x0b,0x0b,0x0b,0x0b";  /* vecteur */


/*-----------------------------------------------------------*/
/*      Declaration de Fonctions                             */
/*-----------------------------------------------------------*/

void des64(unsigned int b1,unsigned int b2,unsigned int c1,unsigned int c2,unsigned int *ps1,unsigned int *ps2);
void invdes64(unsigned int b1,unsigned int b2,unsigned int c1,unsigned int c2,unsigned int *ps1,unsigned int *ps2);
short calcles64(unsigned int c1,unsigned int c2);
static void permut1v64(unsigned int pe1,unsigned int pe2,unsigned int *ps1,unsigned int *ps2);
static void permut2v64(unsigned int pe1,unsigned int pe2,unsigned int *ps1,unsigned int *ps2);
static void initperm64(unsigned int pe1,unsigned int pe2,unsigned int *ps1,unsigned int *ps2);
static void f64(unsigned int *pe,unsigned int c1,unsigned int c2);
static void expansion64(unsigned int pe,unsigned int *ps1,unsigned int *ps2);
static void substitution64(unsigned int pe1,unsigned int pe2,unsigned int *ps);
static void permutation64(unsigned int *pe);
static void finalperm64(unsigned int pe1,unsigned int pe2,unsigned int *ps1,unsigned int *ps2);
/* Prototypes for 32 bits platform */
void des32(unsigned long b1,unsigned long b2,unsigned long c1,unsigned long c2,unsigned long *ps1,unsigned long *ps2);
void invdes32(unsigned long b1,unsigned long b2,unsigned long c1,unsigned long c2,unsigned long *ps1,unsigned long *ps2);
short calcles32(unsigned long c1,unsigned long c2);
static void permut1v32(unsigned long pe1,unsigned long pe2,unsigned long *ps1,unsigned long *ps2);
static void permut2v32(unsigned long pe1,unsigned long pe2,unsigned long *ps1,unsigned long *ps2);
static void initperm32(unsigned long pe1,unsigned long pe2,unsigned long *ps1,unsigned long *ps2);
static void f32(unsigned long *pe,unsigned long c1,unsigned long c2);
static void expansion32(unsigned long pe,unsigned long *ps1,unsigned long *ps2);
static void substitution32(unsigned long pe1,unsigned long pe2,unsigned long *ps);
static void permutation32(unsigned long *pe);
static void finalperm32(unsigned long pe1,unsigned long pe2,unsigned long *ps1,unsigned long *ps2);
static int Is64or32(void);


/*-----------------------------------------------------------*/
/*          Is64or32()                                       */
/*-----------------------------------------------------------*/
int Is64or32()
{
 if (MAXSHORT>0x7FFF) return 64;else return 32;
}


/*-----------------------------------------------------------*/
/*          DesEncrypt64()                                   */
/*-----------------------------------------------------------*/
void DesEncrypt64(unsigned char* in,unsigned char* key,unsigned char* out)
{
    unsigned int    cra,crb;    /* DES Entry  */
    unsigned int    cha,chb;    /* DES Output */
    unsigned int    cla,clb;    /* DES key    */
    int i;              /* Item Loop  */

    /* 1. Convert to type4 */
    for(i=4,cra=0; i>0; i--)
        cra=(cra<<8)+*in++;
    for(i=4,crb=0; i>0; i--)
        crb=(crb<<8)+*in++;
    for(i=4,cla=0; i>0; i--)
        cla=(cla<<8)+*key++;
    for(i=4,clb=0; i>0; i--)
        clb=(clb<<8)+*key++;

/*  printf("DesEncrypt\n");
    printf("cra = %08x\n",cra);
    printf("crb = %08x\n",crb);
    printf("cla = %08x\n",cla);
    printf("clb = %08x\n",clb); */
    /* 2. Encrypt */
    des64(cra,crb,cla,clb,&cha,&chb);

    /* 3. Convert to unsigned char */
/*  printf("cha = %08x\n",cha);
    printf("chb = %08x\n",chb); */
    *out++=(unsigned char)(cha>>24);
    *out++=(unsigned char)(cha>>16);
    *out++=(unsigned char)(cha>>8);
    *out++=(unsigned char)cha;
    *out++=(unsigned char)(chb>>24);
    *out++=(unsigned char)(chb>>16);
    *out++=(unsigned char)(chb>>8);
    *out=(unsigned char)chb;
}


/*-----------------------------------------------------------*/
/*          DesEncrypt32()                                   */
/*-----------------------------------------------------------*/
void DesEncrypt32(unsigned char* in,unsigned char* key,unsigned char* out)
{
    unsigned long   cra,crb;    /* DES Entry  */
    unsigned long   cha,chb;    /* DES Output */
    unsigned long   cla,clb;    /* DES key    */
    int i;              /* Item Loop  */

    /* 1. Convert to type4 */
    for(i=4,cra=0; i>0; i--)
        cra=(cra<<8)+*in++;
    for(i=4,crb=0; i>0; i--)
        crb=(crb<<8)+*in++;
    for(i=4,cla=0; i>0; i--)
        cla=(cla<<8)+*key++;
    for(i=4,clb=0; i>0; i--)
        clb=(clb<<8)+*key++;

/*  printf("DesEncrypt\n");
    printf("cra = %08x\n",cra);
    printf("crb = %08x\n",crb);
    printf("cla = %08x\n",cla);
    printf("clb = %08x\n",clb); */
    /* 2. Encrypt */
    des32(cra,crb,cla,clb,&cha,&chb);

    /* 3. Convert to unsigned char */
/*  printf("cha = %08x\n",cha);
    printf("chb = %08x\n",chb); */
    *out++=(unsigned char)(cha>>24);
    *out++=(unsigned char)(cha>>16);
    *out++=(unsigned char)(cha>>8);
    *out++=(unsigned char)cha;
    *out++=(unsigned char)(chb>>24);
    *out++=(unsigned char)(chb>>16);
    *out++=(unsigned char)(chb>>8);
    *out=(unsigned char)chb;
}


/*-----------------------------------------------------------*/
/*          DesDecrypt64()                                   */
/*-----------------------------------------------------------*/
void DesDecrypt64(unsigned char* in,unsigned char* key,unsigned char* out)
{
    unsigned int    cra,crb;    /* DES Entry  */
    unsigned int    cha,chb;    /* DES Output */
    unsigned int    cla,clb;    /* DES key    */
    int     i;          /* Item Loop  */

    /* 1. Convert to type4 */
    for(i=4,cra=0; i>0; i--)
        cra=(cra<<8)+*in++;
    for(i=4,crb=0; i>0; i--)
        crb=(crb<<8)+*in++;
    for(i=4,cla=0; i>0; i--)
        cla=(cla<<8)+*key++;
    for(i=4,clb=0; i>0; i--)
        clb=(clb<<8)+*key++;

    /* 2. Encrypt */
    invdes64(cra,crb,cla,clb,&cha,&chb);

    /* 3. Convert to unsigned char */
    *out++=(unsigned char)(cha>>24);
    *out++=(unsigned char)(cha>>16);
    *out++=(unsigned char)(cha>>8);
    *out++=(unsigned char)cha;
    *out++=(unsigned char)(chb>>24);
    *out++=(unsigned char)(chb>>16);
    *out++=(unsigned char)(chb>>8);
    *out=(unsigned char)chb;
}


/*-----------------------------------------------------------*/
/*          DesDecrypt32()                                   */
/*-----------------------------------------------------------*/
void DesDecrypt32(unsigned char* in,unsigned char* key,unsigned char* out)
{
    unsigned long   cra,crb;    /* DES Entry  */
    unsigned long   cha,chb;    /* DES Output */
    unsigned long   cla,clb;    /* DES key    */
    int     i;          /* Item Loop  */

    /* 1. Convert to type4 */
    for(i=4,cra=0; i>0; i--)
        cra=(cra<<8)+*in++;
    for(i=4,crb=0; i>0; i--)
        crb=(crb<<8)+*in++;
    for(i=4,cla=0; i>0; i--)
        cla=(cla<<8)+*key++;
    for(i=4,clb=0; i>0; i--)
        clb=(clb<<8)+*key++;

    /* 2. Encrypt */
    invdes32(cra,crb,cla,clb,&cha,&chb);

    /* 3. Convert to unsigned char */
    *out++=(unsigned char)(cha>>24);
    *out++=(unsigned char)(cha>>16);
    *out++=(unsigned char)(cha>>8);
    *out++=(unsigned char)cha;
    *out++=(unsigned char)(chb>>24);
    *out++=(unsigned char)(chb>>16);
    *out++=(unsigned char)(chb>>8);
    *out=(unsigned char)chb;
}

/*-----------------------------------------------------------*/
/*          DecryptCBC()                                     */
/*-----------------------------------------------------------*/
/*  Cipher Bloc Chaining Decryption, with IV  */
void DecryptCBC(unsigned char* cle,unsigned char* msg,int lng)
{
    int     i;      /* Item Loop        */
    unsigned char   vct[8]; /* Vector           */
    unsigned char   ivs[8]; /* Initial Vector   */

    memcpy(ivs,vect,8);
    for(; lng>0; lng-=8,msg+=8){
        memcpy(vct,msg,8);
        if (Is64or32()==32) DesDecrypt32(msg,cle,msg);
        else DesDecrypt64(msg,cle,msg);
        for(i=0; (i<lng)&&(i<8); i++)
            *(msg+i)^=*(ivs+i);
        memcpy(ivs,vct,8);
    }
}


/*-----------------------------------------------------------*/
/*          EncryptCBC()                                     */
/*-----------------------------------------------------------*/
/* Cipher Bloc Chaining Encryption, with IV  */
void EncryptCBC(unsigned char* cle,unsigned char* msg,int lng)
{
    int     i;      /* Item Loop */
    unsigned char   vct[8]; /* Vector    */

    memcpy(vct,vect,8);
    for(; lng>0; lng-=8,msg+=8){
        for(i=0; (i<lng)&&(i<8); i++)
            *(msg+i)^=*(vct+i);
        if (Is64or32()==32) DesEncrypt32(msg,cle,msg);
        else DesEncrypt64(msg,cle,msg);
        memcpy(vct,msg,8);
    }
}


/*-----------------------------------------------------------*/
/*          TDesEncrypt()                                    */
/*-----------------------------------------------------------*/
/*  Triple DES Encryption  */
void TDesEncrypt(unsigned char* in,unsigned char* key,unsigned char* out)
{
    if (Is64or32()==32) {
        DesEncrypt32(in,key,out);
        DesDecrypt32(out,key+8,out);
        DesEncrypt32(out,key,out);
   }
   else {
    DesEncrypt64(in,key,out);
        DesDecrypt64(out,key+8,out);
        DesEncrypt64(out,key,out);
   }
}


/*-----------------------------------------------------------*/
/*          TDesDecrypt()                                    */
/*-----------------------------------------------------------*/
/*  Triple DES Decryption  */
void TDesDecrypt(unsigned char* in,unsigned char* key,unsigned char* out)
{
    if (Is64or32()==32) {
        DesDecrypt32(in,key,out);
        DesEncrypt32(out,key+8,out);
        DesDecrypt32(out,key,out);
   }
   else {
    DesDecrypt64(in,key,out);
        DesEncrypt64(out,key+8,out);
        DesDecrypt64(out,key,out);
  }
}


/*-----------------------------------------------------------*/
/*          TDesDecryptCBC()                                 */
/*-----------------------------------------------------------*/
/*  Triple DES Cipher Bloc Chaining Decryption, with IV  */
void TDesDecryptCBC(unsigned char* cle,unsigned char* msg,int lng,unsigned char* iv)
{
    int     i;      /* Item Loop        */
    unsigned char   vct[8]; /* Vector           */
    unsigned char   ivs[8]; /* Initial Vector   */

    memcpy(ivs,iv,8);
    for(; lng>0; lng-=8,msg+=8){
        memcpy(vct,msg,8);
        TDesDecrypt(msg,cle,msg);
        for(i=0; (i<lng)&&(i<8); i++)
            *(msg+i)^=*(ivs+i);
        memcpy(ivs,vct,8);
    }
}


/*-----------------------------------------------------------*/
/*          TDesEncryptCBC()                                 */
/*-----------------------------------------------------------*/
/*  Triple DES Cipher Bloc Chaining Encryption, with IV  */
void TDesEncryptCBC(unsigned char* cle,unsigned char* msg,int lng,unsigned char* iv)
{
    int     i;      /* Item Loop */
    unsigned char   vct[8]; /* Vector    */

    memcpy(vct,iv,8);
    for(; lng>0; lng-=8,msg+=8){
        for(i=0; (i<lng)&&(i<8); i++)
            *(msg+i)^=*(vct+i);
        TDesEncrypt(msg,cle,msg);
        memcpy(vct,msg,8);
    }
}


/*-----------------------------------------------------------*/
/*            des64()                                        */
/*-----------------------------------------------------------*/
/*  Chiffrement d'un bloc de 64 bits par une cle de 64 bits
/*
/*  Entree  :   b1b2    (bloc de 64 bits)
/*              c1c2    (bloc de 64 bits)
/*  Sortie  :   s1s2    (bloc chiffre de 64 bits)
/**/
void des64(unsigned int b1,unsigned int b2,unsigned int c1,unsigned int c2,unsigned int *ps1,unsigned int *ps2)
{
    unsigned int     l, r, x;
    short   i;

    calcles64(c1,c2);                 /* calcul des 16 cles intermediaires        */
    initperm64(b1,b2,&l,&r);          /* permutation initiale                     */

    for (i=0; i<16; i++) {            /* 16 iterations                            */
        x= l; l= r;                   /* echanger demis-bloc gauche et droit      */
        f64(&r,c64[0][i],c64[1][i]);  /* fonction f du bloc droit et de la cle    */
        r^= x;                        /* ou exclusif bloc gauche et resultat de f */
    }

    finalperm64(r,l,ps1,ps2);         /* permutation finale                       */
}


/*-----------------------------------------------------------*/
/*            des32()                                        */
/*-----------------------------------------------------------*/
/*  Chiffrement d'un bloc de 64 bits par une cle de 64 bits
/*
/*  Entree  :   b1b2    (bloc de 64 bits)
/*              c1c2    (bloc de 64 bits)
/*  Sortie  :   s1s2    (bloc chiffre de 64 bits)
/**/
void des32(unsigned long b1,unsigned long b2,unsigned long c1,unsigned long c2,unsigned long *ps1,unsigned long *ps2)
{
    unsigned long    l, r, x;
    short   i;

    calcles32(c1,c2);                 /* calcul des 16 cles intermediaires        */
    initperm32(b1,b2,&l,&r);          /* permutation initiale                     */

    for (i=0; i<16; i++) {            /* 16 iterations                            */
        x= l; l= r;                   /* echanger demis-bloc gauche et droit      */
        f32(&r,c32[0][i],c32[1][i]);  /* fonction f du bloc droit et de la cle    */
        r^= x;                        /* ou exclusif bloc gauche et resultat de f */
    }

    finalperm32(r,l,ps1,ps2);         /* permutation finale                       */
}


/*-----------------------------------------------------------*/
/*            invdes64()                                     */
/*-----------------------------------------------------------*/
/*  Dechiffrement d'un bloc de 64 bits par une cle de 64 bits
/*
/*  Entree  :   b1b2    (bloc chiffre de 64 bits)
/*              c1c2    (bloc de 64 bits)
/*  Sortie  :   s1s2    (bloc en clair de 64 bits)
/*
/**/
void invdes64(unsigned int b1,unsigned int b2,unsigned int c1,unsigned int c2,unsigned int *ps1,unsigned int *ps2)
{
    unsigned int     l, r, x;
    short   i;

    calcles64(c1,c2);                 /* calcul des 16 cles intermediaires        */
    initperm64(b1,b2,&l,&r);          /* permutation initiale                     */

    for (i=0; i<16; i++) {            /* 16 iterations                            */
        x= l; l= r;                   /* echanger demis-bloc gauche et droit      */
        f64(&r,c64[0][15-i],          /* fonction f du bloc droit et de la cle    */
        c64[1][15-i]);
        r^= x;                        /* ou exclusif bloc gauche et resultat de f */
    }

    finalperm64(r,l,ps1,ps2);         /* permutation finale                       */
}


/*-----------------------------------------------------------*/
/*            invdes32()                                     */
/*-----------------------------------------------------------*/
/*  Dechiffrement d'un bloc de 64 bits par une cle de 64 bits
/*
/*  Entree  :   b1b2    (bloc chiffre de 64 bits)
/*              c1c2    (bloc de 64 bits)
/*  Sortie  :   s1s2    (bloc en clair de 64 bits)
/**/
void invdes32(unsigned long b1,unsigned long b2,unsigned long c1,unsigned long c2,unsigned long *ps1,unsigned long *ps2)
{
    unsigned long    l, r, x;
    short   i;

    calcles32(c1,c2);                 /* calcul des 16 cles intermediaires        */
    initperm32(b1,b2,&l,&r);          /* permutation initiale                     */

    for (i=0; i<16; i++) {            /* 16 iterations                            */
        x= l; l= r;                   /* echanger demis-bloc gauche et droit      */
        f32(&r,c32[0][15-i],          /* fonction f du bloc droit et de la cle    */
        c32[1][15-i]);
        r^= x;                        /* ou exclusif bloc gauche et resultat de f */
    }

    finalperm32(r,l,ps1,ps2);         /* permutation finale                       */
}


/*-----------------------------------------------------------*/
/*          calcles64()                                      */
/*-----------------------------------------------------------*/
/*  Calcul des 16 cles a partir de la cle initiale
/*
/*  Entree  :   c1c2            (64 bits, 1 a 32, 33 a 64, parite paire)
/*  Sortie  :   c[1][i]c[2][i]  (48 bits, 1 24,25 a 48, cadre a gauche, i= 0 a 15)
/*
/*  Permutation1 : c1-c2 (64 bits) --> bc-bd (56 bits, cadre a gauche)
/*                 Pour i= 1 a 16
/*                 Decalage gauche bc
/*                 Decalage gauche bd
/*  Permutation2 : bc-bd (56 bits) --> c[0][i]c[1][i] (48 bits, cadre a gauche)
/**/
short calcles64(unsigned int c1,unsigned int c2)
{
    register short  i;
    unsigned int    bc,bd;

    permut1v64(c1,c2,(unsigned int*)&bc,(unsigned int*)&bd);

    bc= ((bc>>27)|(bc<<1))&M28;
    bd= ((bd>>27)|(bd<<1))&M28;
    permut2v64(bc,bd,&c64[0][0],&c64[1][0]);

    bc= ((bc>>27)|(bc<<1))&M28;
    bd= ((bd>>27)|(bd<<1))&M28;
    permut2v64(bc,bd,&c64[0][1],&c64[1][1]);

    for (i=2; i<8; i++) {
        bc= ((bc>>26)|(bc<<2))&M28;
        bd= ((bd>>26)|(bd<<2))&M28;
        permut2v64(bc,bd,&c64[0][i],&c64[1][i]);
    }

    bc= ((bc>>27)|(bc<<1))&M28;
    bd= ((bd>>27)|(bd<<1))&M28;
    permut2v64(bc,bd,&c64[0][i],&c64[1][i]);

    for (i++; i<15; i++) {
        bc= ((bc>>26)|(bc<<2))&M28;
        bd= ((bd>>26)|(bd<<2))&M28;
        permut2v64(bc,bd,&c64[0][i],&c64[1][i]);
    }

     bc= ((bc>>27)|(bc<<1))&M28;
     bd= ((bd>>27)|(bd<<1))&M28;
    permut2v64(bc,bd,&c64[0][i],&c64[1][i]);

    return(0);
}


/*-----------------------------------------------------------*/
/*          calcles32()                                      */
/*-----------------------------------------------------------*/
/*  Calcul des 16 cles a partir de la cle initiale
/*
/*  Entree  :   c1c2            (64 bits, 1 a 32, 33 a 64, parite paire)
/*  Sortie  :   c[1][i]c[2][i]  (48 bits, 1 24,25 a 48, cadre a gauche, i= 0 a 15)
/*
/*  Permutation1 : c1-c2 (64 bits) --> bc-bd (56 bits, cadre a gauche)
/*                 Pour i= 1 a 16
/*                 Decalage gauche bc
/*                 Decalage gauche bd
/*  Permutation2 : bc-bd (56 bits) --> c[0][i]c[1][i] (48 bits, cadre a gauche)
/**/
short calcles32(unsigned long c1,unsigned long c2)
{
    register short  i;
    unsigned long    bc,bd;

    permut1v32(c1,c2,(unsigned long*)&bc,(unsigned long*)&bd);

    bc= ((bc>>27)|(bc<<1))&M28;
    bd= ((bd>>27)|(bd<<1))&M28;
    permut2v32(bc,bd,&c32[0][0],&c32[1][0]);

    bc= ((bc>>27)|(bc<<1))&M28;
    bd= ((bd>>27)|(bd<<1))&M28;
    permut2v32(bc,bd,&c32[0][1],&c32[1][1]);

    for (i=2; i<8; i++) {
        bc= ((bc>>26)|(bc<<2))&M28;
        bd= ((bd>>26)|(bd<<2))&M28;
        permut2v32(bc,bd,&c32[0][i],&c32[1][i]);
    }

    bc= ((bc>>27)|(bc<<1))&M28;
    bd= ((bd>>27)|(bd<<1))&M28;
    permut2v32(bc,bd,&c32[0][i],&c32[1][i]);

    for (i++; i<15; i++) {
        bc= ((bc>>26)|(bc<<2))&M28;
        bd= ((bd>>26)|(bd<<2))&M28;
        permut2v32(bc,bd,&c32[0][i],&c32[1][i]);
    }

    bc= ((bc>>27)|(bc<<1))&M28;
    bd= ((bd>>27)|(bd<<1))&M28;
    permut2v32(bc,bd,&c32[0][i],&c32[1][i]);

    return(0);
}

/*-----------------------------------------------------------*/
/*          permut1v64()                                     */
/*-----------------------------------------------------------*/
/*  Permutation Cle - choix 1
/*
/*  Entree  :   e1e2 (64 bits, 1 a 32, 33 a 64)
/*  Sortie  :   s1s2 (56 bits, 1 a 28, 29 a 56, cadre a gauche)
/*
/*  t1...t64 --> t57...t4
/*  s:1 (s1:1)    =  e:57 (e2:25)     <==>  si e2:25=1   alors s1:1=1
/*                ...........
/*  s:56 (s2:28)  =  e:4  (e1:4)      <==>  si e1:4=1    alors s2:28=1
/**/
static void permut1v64(unsigned int pe1,unsigned int pe2,unsigned int *ps1,unsigned int *ps2)
{
   register unsigned int    e1, e2, s1=0, s2=0;

   e1= pe1; e2= pe2;

   if (e2&B25)  s1|=B01;    if (e2&B17)  s1|=B02;
   if (e2&B09)  s1|=B03;    if (e2&B01)  s1|=B04;
   if (e2&B26)  s1|=B09;    if (e2&B18)  s1|=B10;
   if (e2&B10)  s1|=B11;    if (e2&B02)  s1|=B12;
   if (e2&B27)  s1|=B17;    if (e2&B19)  s1|=B18;
   if (e2&B11)  s1|=B19;    if (e2&B03)  s1|=B20;
   if (e2&B28)  s1|=B25;    if (e2&B20)  s1|=B26;
   if (e2&B12)  s1|=B27;    if (e2&B04)  s1|=B28;

   if (e2&B31)  s2|=B01;    if (e2&B23)  s2|=B02;
   if (e2&B15)  s2|=B03;    if (e2&B07)  s2|=B04;
   if (e2&B30)  s2|=B09;    if (e2&B22)  s2|=B10;
   if (e2&B14)  s2|=B11;    if (e2&B06)  s2|=B12;
   if (e2&B29)  s2|=B17;    if (e2&B21)  s2|=B18;
   if (e2&B13)  s2|=B19;    if (e2&B05)  s2|=B20;

   if (e1&B31)  s2|=B05;    if (e1&B23)  s2|=B06;
   if (e1&B15)  s2|=B07;    if (e1&B07)  s2|=B08;
   if (e1&B30)  s2|=B13;    if (e1&B22)  s2|=B14;
   if (e1&B14)  s2|=B15;    if (e1&B06)  s2|=B16;
   if (e1&B29)  s2|=B21;    if (e1&B21)  s2|=B22;
   if (e1&B13)  s2|=B23;    if (e1&B05)  s2|=B24;
   if (e1&B28)  s2|=B25;    if (e1&B20)  s2|=B26;
   if (e1&B12)  s2|=B27;    if (e1&B04)  s2|=B28;

   if (e1&B25)  s1|=B05;    if (e1&B17)  s1|=B06;
   if (e1&B09)  s1|=B07;    if (e1&B01)  s1|=B08;
   if (e1&B26)  s1|=B13;    if (e1&B18)  s1|=B14;
   if (e1&B10)  s1|=B15;    if (e1&B02)  s1|=B16;
   if (e1&B27)  s1|=B21;    if (e1&B19)  s1|=B22;
   if (e1&B11)  s1|=B23;    if (e1&B03)  s1|=B24;

   *ps1= s1; *ps2= s2;
}


/*-----------------------------------------------------------*/
/*          permut1v32()                                     */
/*-----------------------------------------------------------*/
/*  Permutation Cle - choix 1
/*
/*  Entree  :   e1e2 (64 bits, 1 a 32, 33 a 64)
/*  Sortie  :   s1s2 (56 bits, 1 a 28, 29 a 56, cadre a gauche)
/*
/*  t1...t64 --> t57...t4
/*  s:1 (s1:1)    =  e:57 (e2:25)     <==>  si e2:25=1   alors s1:1=1
/*                ...........
/*  s:56 (s2:28)  =  e:4  (e1:4)      <==>  si e1:4=1    alors s2:28=1
/**/
static void permut1v32(unsigned long pe1,unsigned long pe2,unsigned long *ps1,unsigned long *ps2)
{
   register unsigned long   e1, e2, s1=0, s2=0;

   e1= pe1; e2= pe2;

   if (e2&B25)  s1|=B01;    if (e2&B17)  s1|=B02;
   if (e2&B09)  s1|=B03;    if (e2&B01)  s1|=B04;
   if (e2&B26)  s1|=B09;    if (e2&B18)  s1|=B10;
   if (e2&B10)  s1|=B11;    if (e2&B02)  s1|=B12;
   if (e2&B27)  s1|=B17;    if (e2&B19)  s1|=B18;
   if (e2&B11)  s1|=B19;    if (e2&B03)  s1|=B20;
   if (e2&B28)  s1|=B25;    if (e2&B20)  s1|=B26;
   if (e2&B12)  s1|=B27;    if (e2&B04)  s1|=B28;

   if (e2&B31)  s2|=B01;    if (e2&B23)  s2|=B02;
   if (e2&B15)  s2|=B03;    if (e2&B07)  s2|=B04;
   if (e2&B30)  s2|=B09;    if (e2&B22)  s2|=B10;
   if (e2&B14)  s2|=B11;    if (e2&B06)  s2|=B12;
   if (e2&B29)  s2|=B17;    if (e2&B21)  s2|=B18;
   if (e2&B13)  s2|=B19;    if (e2&B05)  s2|=B20;

   if (e1&B31)  s2|=B05;    if (e1&B23)  s2|=B06;
   if (e1&B15)  s2|=B07;    if (e1&B07)  s2|=B08;
   if (e1&B30)  s2|=B13;    if (e1&B22)  s2|=B14;
   if (e1&B14)  s2|=B15;    if (e1&B06)  s2|=B16;
   if (e1&B29)  s2|=B21;    if (e1&B21)  s2|=B22;
   if (e1&B13)  s2|=B23;    if (e1&B05)  s2|=B24;
   if (e1&B28)  s2|=B25;    if (e1&B20)  s2|=B26;
   if (e1&B12)  s2|=B27;    if (e1&B04)  s2|=B28;

   if (e1&B25)  s1|=B05;    if (e1&B17)  s1|=B06;
   if (e1&B09)  s1|=B07;    if (e1&B01)  s1|=B08;
   if (e1&B26)  s1|=B13;    if (e1&B18)  s1|=B14;
   if (e1&B10)  s1|=B15;    if (e1&B02)  s1|=B16;
   if (e1&B27)  s1|=B21;    if (e1&B19)  s1|=B22;
   if (e1&B11)  s1|=B23;    if (e1&B03)  s1|=B24;

   *ps1= s1; *ps2= s2;
}

/*-----------------------------------------------------------*/
/*          permut2v64()                                     */
/*-----------------------------------------------------------*/
/*  Permutation Cle - choix 2
/*
/*  Entree  :   e1e2 (56 bits, 1 a 28, 29 a 56, cadre a gauche)
/*  Sortie  :   s1s2 (48 bits, 1 a 24, 25 a 48, cadre a gauche)
/*
/*  t1...t56 --> t14...t32
/*  s:1 (s1:1)    =  e:14 (e1:14)     <==>  si e1:14=1   alors s1:1=1
/*                ...........
/*  s:48 (s2:24)  =  e:32 (e2:4)      <==>  si e2:4=1    alors s2:24=1
/**/
static void permut2v64(unsigned int pe1,unsigned int pe2,unsigned int *ps1,unsigned int *ps2)
{
   register unsigned int    e1, e2, s1=0, s2=0;

   e1= pe1; e2= pe2;

   if (e1&B14)  s1|=B01;   if (e1&B17)  s1|=B02;
   if (e1&B11)  s1|=B03;   if (e1&B24)  s1|=B04;
   if (e1&B01)  s1|=B05;   if (e1&B05)  s1|=B06;
   if (e1&B03)  s1|=B07;   if (e1&B28)  s1|=B08;
   if (e1&B15)  s1|=B09;   if (e1&B06)  s1|=B10;
   if (e1&B21)  s1|=B11;   if (e1&B10)  s1|=B12;
   if (e1&B23)  s1|=B13;   if (e1&B19)  s1|=B14;
   if (e1&B12)  s1|=B15;   if (e1&B04)  s1|=B16;
   if (e1&B26)  s1|=B17;   if (e1&B08)  s1|=B18;
   if (e1&B16)  s1|=B19;   if (e1&B07)  s1|=B20;
   if (e1&B27)  s1|=B21;   if (e1&B20)  s1|=B22;
   if (e1&B13)  s1|=B23;   if (e1&B02)  s1|=B24;

   if (e2&B13)  s2|=B01;   if (e2&B24)  s2|=B02;
   if (e2&B03)  s2|=B03;   if (e2&B09)  s2|=B04;
   if (e2&B19)  s2|=B05;   if (e2&B27)  s2|=B06;
   if (e2&B02)  s2|=B07;   if (e2&B12)  s2|=B08;
   if (e2&B23)  s2|=B09;   if (e2&B17)  s2|=B10;
   if (e2&B05)  s2|=B11;   if (e2&B20)  s2|=B12;
   if (e2&B16)  s2|=B13;   if (e2&B21)  s2|=B14;
   if (e2&B11)  s2|=B15;   if (e2&B28)  s2|=B16;
   if (e2&B06)  s2|=B17;   if (e2&B25)  s2|=B18;
   if (e2&B18)  s2|=B19;   if (e2&B14)  s2|=B20;
   if (e2&B22)  s2|=B21;   if (e2&B08)  s2|=B22;
   if (e2&B01)  s2|=B23;   if (e2&B04)  s2|=B24;

   *ps1= s1; *ps2= s2;
}


/*-----------------------------------------------------------*/
/*          permut2v32()                                     */
/*-----------------------------------------------------------*/
/*  Permutation Cle - choix 2
/*
/*  Entree  :   e1e2 (56 bits, 1 a 28, 29 a 56, cadre a gauche)
/*  Sortie  :   s1s2 (48 bits, 1 a 24, 25 a 48, cadre a gauche)
/*
/*  t1...t56 --> t14...t32
/*  s:1 (s1:1)    =  e:14 (e1:14)     <==>  si e1:14=1   alors s1:1=1
/*                ...........
/*  s:48 (s2:24)  =  e:32 (e2:4)      <==>  si e2:4=1    alors s2:24=1
/**/
static void permut2v32(unsigned long pe1,unsigned long pe2,unsigned long *ps1,unsigned long *ps2)
{
   register unsigned long   e1, e2, s1=0, s2=0;

   e1= pe1; e2= pe2;

   if (e1&B14)  s1|=B01;   if (e1&B17)  s1|=B02;
   if (e1&B11)  s1|=B03;   if (e1&B24)  s1|=B04;
   if (e1&B01)  s1|=B05;   if (e1&B05)  s1|=B06;
   if (e1&B03)  s1|=B07;   if (e1&B28)  s1|=B08;
   if (e1&B15)  s1|=B09;   if (e1&B06)  s1|=B10;
   if (e1&B21)  s1|=B11;   if (e1&B10)  s1|=B12;
   if (e1&B23)  s1|=B13;   if (e1&B19)  s1|=B14;
   if (e1&B12)  s1|=B15;   if (e1&B04)  s1|=B16;
   if (e1&B26)  s1|=B17;   if (e1&B08)  s1|=B18;
   if (e1&B16)  s1|=B19;   if (e1&B07)  s1|=B20;
   if (e1&B27)  s1|=B21;   if (e1&B20)  s1|=B22;
   if (e1&B13)  s1|=B23;   if (e1&B02)  s1|=B24;

   if (e2&B13)  s2|=B01;   if (e2&B24)  s2|=B02;
   if (e2&B03)  s2|=B03;   if (e2&B09)  s2|=B04;
   if (e2&B19)  s2|=B05;   if (e2&B27)  s2|=B06;
   if (e2&B02)  s2|=B07;   if (e2&B12)  s2|=B08;
   if (e2&B23)  s2|=B09;   if (e2&B17)  s2|=B10;
   if (e2&B05)  s2|=B11;   if (e2&B20)  s2|=B12;
   if (e2&B16)  s2|=B13;   if (e2&B21)  s2|=B14;
   if (e2&B11)  s2|=B15;   if (e2&B28)  s2|=B16;
   if (e2&B06)  s2|=B17;   if (e2&B25)  s2|=B18;
   if (e2&B18)  s2|=B19;   if (e2&B14)  s2|=B20;
   if (e2&B22)  s2|=B21;   if (e2&B08)  s2|=B22;
   if (e2&B01)  s2|=B23;   if (e2&B04)  s2|=B24;

    *ps1= s1; *ps2= s2;
}

/*-----------------------------------------------------------*/
/*          initperm64()                                     */
/*-----------------------------------------------------------*/
/*  Permutation Initiale
/*
/*  Entree  :   e1e2 (64 bits, 1 a 32, 33 a 64)
/*  Sortie  :   s1s2 (64 bits, 1 a 32, 33 a 64)
/*
/*  t1t2...t64 --> t58t50...t7
/*  s:1 (s1:1)    =  e:58 (e2:26)  <==>  si e2:26=1  alors s1:1=1
/*  s:2 (s1:2)    =  e:50 (e2:18)  <==>  si e2:18=1  alors s1:2=1
/*              ...........
/*  s:64 (s2:32)  =  e:7 (e1:7)    <==>  si e2:26=1  alors s2:32=1
/**/
static void initperm64(unsigned int pe1,unsigned int pe2,unsigned int *ps1,unsigned int *ps2)
{
   register unsigned int    e1, e2, s1=0, s2=0;

   e1= pe1; e2= pe2;

   if (e2&B26)  s1|=B01;   if (e2&B18)  s1|=B02;
   if (e2&B10)  s1|=B03;   if (e2&B02)  s1|=B04;
   if (e2&B28)  s1|=B09;   if (e2&B20)  s1|=B10;
   if (e2&B12)  s1|=B11;   if (e2&B04)  s1|=B12;
   if (e2&B30)  s1|=B17;   if (e2&B22)  s1|=B18;
   if (e2&B14)  s1|=B19;   if (e2&B06)  s1|=B20;
   if (e2&B32)  s1|=B25;   if (e2&B24)  s1|=B26;
   if (e2&B16)  s1|=B27;   if (e2&B08)  s1|=B28;

   if (e2&B25)  s2|=B01;   if (e2&B17)  s2|=B02;
   if (e2&B09)  s2|=B03;   if (e2&B01)  s2|=B04;
   if (e2&B27)  s2|=B09;   if (e2&B19)  s2|=B10;
   if (e2&B11)  s2|=B11;   if (e2&B03)  s2|=B12;
   if (e2&B29)  s2|=B17;   if (e2&B21)  s2|=B18;
   if (e2&B13)  s2|=B19;   if (e2&B05)  s2|=B20;
   if (e2&B31)  s2|=B25;   if (e2&B23)  s2|=B26;
   if (e2&B15)  s2|=B27;   if (e2&B07)  s2|=B28;

   if (e1&B25)  s2|=B05;   if (e1&B17)  s2|=B06;
   if (e1&B09)  s2|=B07;   if (e1&B01)  s2|=B08;
   if (e1&B27)  s2|=B13;   if (e1&B19)  s2|=B14;
   if (e1&B11)  s2|=B15;   if (e1&B03)  s2|=B16;
   if (e1&B29)  s2|=B21;   if (e1&B21)  s2|=B22;
   if (e1&B13)  s2|=B23;   if (e1&B05)  s2|=B24;
   if (e1&B31)  s2|=B29;   if (e1&B23)  s2|=B30;
   if (e1&B15)  s2|=B31;   if (e1&B07)  s2|=B32;

   if (e1&B26)  s1|=B05;   if (e1&B18)  s1|=B06;
   if (e1&B10)  s1|=B07;   if (e1&B02)  s1|=B08;
   if (e1&B28)  s1|=B13;   if (e1&B20)  s1|=B14;
   if (e1&B12)  s1|=B15;   if (e1&B04)  s1|=B16;
   if (e1&B30)  s1|=B21;   if (e1&B22)  s1|=B22;
   if (e1&B14)  s1|=B23;   if (e1&B06)  s1|=B24;
   if (e1&B32)  s1|=B29;   if (e1&B24)  s1|=B30;
   if (e1&B16)  s1|=B31;   if (e1&B08)  s1|=B32;

   *ps1= s1; *ps2= s2;
}


/*-----------------------------------------------------------*/
/*          initperm32()                                     */
/*-----------------------------------------------------------*/
/*  Permutation Initiale
/*
/*  Entree  :   e1e2 (64 bits, 1 a 32, 33 a 64)
/*  Sortie  :   s1s2 (64 bits, 1 a 32, 33 a 64)
/*
/*  t1t2...t64 --> t58t50...t7
/*  s:1 (s1:1)    =  e:58 (e2:26)  <==>  si e2:26=1  alors s1:1=1
/*  s:2 (s1:2)    =  e:50 (e2:18)  <==>  si e2:18=1  alors s1:2=1
/*              ...........
/*  s:64 (s2:32)  =  e:7 (e1:7)    <==>  si e2:26=1  alors s2:32=1
/**/
static void initperm32(unsigned long pe1,unsigned long pe2,unsigned long *ps1,unsigned long *ps2)
{
   register unsigned long   e1, e2, s1=0, s2=0;

   e1= pe1; e2= pe2;

   if (e2&B26)  s1|=B01;   if (e2&B18)  s1|=B02;
   if (e2&B10)  s1|=B03;   if (e2&B02)  s1|=B04;
   if (e2&B28)  s1|=B09;   if (e2&B20)  s1|=B10;
   if (e2&B12)  s1|=B11;   if (e2&B04)  s1|=B12;
   if (e2&B30)  s1|=B17;   if (e2&B22)  s1|=B18;
   if (e2&B14)  s1|=B19;   if (e2&B06)  s1|=B20;
   if (e2&B32)  s1|=B25;   if (e2&B24)  s1|=B26;
   if (e2&B16)  s1|=B27;   if (e2&B08)  s1|=B28;

   if (e2&B25)  s2|=B01;   if (e2&B17)  s2|=B02;
   if (e2&B09)  s2|=B03;   if (e2&B01)  s2|=B04;
   if (e2&B27)  s2|=B09;   if (e2&B19)  s2|=B10;
   if (e2&B11)  s2|=B11;   if (e2&B03)  s2|=B12;
   if (e2&B29)  s2|=B17;   if (e2&B21)  s2|=B18;
   if (e2&B13)  s2|=B19;   if (e2&B05)  s2|=B20;
   if (e2&B31)  s2|=B25;   if (e2&B23)  s2|=B26;
   if (e2&B15)  s2|=B27;   if (e2&B07)  s2|=B28;

   if (e1&B25)  s2|=B05;   if (e1&B17)  s2|=B06;
   if (e1&B09)  s2|=B07;   if (e1&B01)  s2|=B08;
   if (e1&B27)  s2|=B13;   if (e1&B19)  s2|=B14;
   if (e1&B11)  s2|=B15;   if (e1&B03)  s2|=B16;
   if (e1&B29)  s2|=B21;   if (e1&B21)  s2|=B22;
   if (e1&B13)  s2|=B23;   if (e1&B05)  s2|=B24;
   if (e1&B31)  s2|=B29;   if (e1&B23)  s2|=B30;
   if (e1&B15)  s2|=B31;   if (e1&B07)  s2|=B32;

   if (e1&B26)  s1|=B05;   if (e1&B18)  s1|=B06;
   if (e1&B10)  s1|=B07;   if (e1&B02)  s1|=B08;
   if (e1&B28)  s1|=B13;   if (e1&B20)  s1|=B14;
   if (e1&B12)  s1|=B15;   if (e1&B04)  s1|=B16;
   if (e1&B30)  s1|=B21;   if (e1&B22)  s1|=B22;
   if (e1&B14)  s1|=B23;   if (e1&B06)  s1|=B24;
   if (e1&B32)  s1|=B29;   if (e1&B24)  s1|=B30;
   if (e1&B16)  s1|=B31;   if (e1&B08)  s1|=B32;

   *ps1= s1; *ps2= s2;
}

/*-----------------------------------------------------------*/
/*              f64()                                        */
/*-----------------------------------------------------------*/
/*  Fonction de calcul sur un bloc de 32 bits
/*
/*  Entree  :   e     (32 bits)
/*              c1c2  cle intermediaire (48 bits, 1 a 24, 25 a 48)
/*  Sortie  :   e     (32 bits)
/**/
static void f64(unsigned int *pe,unsigned int c1,unsigned int c2)
{
    unsigned int    s1, s2;

    expansion64(*pe,&s1,&s2);
    substitution64(s1^c1,s2^c2,pe);
    permutation64(pe);
}


/*-----------------------------------------------------------*/
/*              f32()                                        */
/*-----------------------------------------------------------*/
/*  Fonction de calcul sur un bloc de 32 bits
/*
/*  Entree  :   e     (32 bits)
/*              c1c2  cle intermediaire (48 bits, 1 a 24, 25 a 48)
/*  Sortie  :   e     (32 bits)
/**/
static void f32(unsigned long *pe,unsigned long c1,unsigned long c2)
{
    unsigned long   s1, s2;

    expansion32(*pe,&s1,&s2);
    substitution32(s1^c1,s2^c2,pe);
    permutation32(pe);
}


/*-----------------------------------------------------------*/
/*          expansion64()                                    */
/*-----------------------------------------------------------*/
/*  Expansion bloc 32 bits en bloc 48 bits
/*
/*  Entree  :   e    (32 bits)
/*  Sortie  :   s1s2 (48 bits, 1 a 24, 25 a 48, cadre a gauche)
/*
/*  On utilise la configuration de bits, en effectuant des decalages 
/*  et en masquant, pour obtenir chaque sous-ensemble de huit bits.
/**/
static void expansion64(unsigned int pe,unsigned int *ps1,unsigned int *ps2)
{
    register unsigned int     e, s1, s2;

    e= pe; s1= 0;           /* initialisation                  */
    if (e&B32)  s1=B01;     /* s:1= e:32                       */

    e>>= 1;                 /* e:1 --> bit2     e:4 --> bit5   */
    s1|= e&(0x7C000000L);   /* e:1-e:5 --> s:2-s:6             */

    e>>= 2;                 /* e:4 --> bit7     e:8 --> bit11  */
    s1|= e&(0x3F00000L);    /* e:4-e:9 --> s:7-s:12            */

    e>>= 2;                 /* e:8 --> bit13    e:12--> bit17  */
    s1|= e&(0xFC000L);      /* e:8-e:13 --> s:13-s:18          */

    e>>= 2;                 /* e:12 --> bit19                  */
    s1|= e&(0x3F00L);       /* e:12-e:17 --> s:19-s:24         */


    e= pe; s2= 0;           /* reinitialisation                */
    if (e&B01)  s2=B24;     /* s:24= e:1                       */

    e<<= 9;                 /* e:32 --> bit23   e:29 --> bit20 */
    s2|= e&(0x3E00L);       /* e:28-e:32 --> s:19-s:23         */

    e<<= 2;                 /* e:29 --> bit18   e:25--> bit14  */
    s2|= e&(0xFC000L);      /* e:24-e:29 --> s:13-s:18         */

    e<<= 2;                 /* e:25 --> bit12   e:21 --> bit8  */
    s2|= e&(0x3F00000L);    /* e:20-e:25 --> s:7-s:12          */

    e<<= 2;                 /* e:21 --> bit6                   */
    s2|= e&(0xFC000000L);   /* e:16-e:20 --> s:1-s:6           */

    *ps1= s1; *ps2= s2;
}


/*-----------------------------------------------------------*/
/*          expansion32()                                    */
/*-----------------------------------------------------------*/
/*  Expansion bloc 32 bits en bloc 48 bits
/*
/*  Entree  :   e    (32 bits)
/*  Sortie  :   s1s2 (48 bits, 1 a 24, 25 a 48, cadre a gauche)
/*
/*  On utilise la configuration de bits, en effectuant des decalages 
/*  et en masquant, pour obtenir chaque sous-ensemble de huit bits.
/**/
static void expansion32(unsigned long pe,unsigned long *ps1,unsigned long *ps2)
{
    register unsigned long    e, s1, s2;

    e= pe; s1= 0;           /* initialisation                   */
    if (e&B32)  s1=B01;     /* s:1= e:32                        */

    e>>= 1;                 /* e:1 --> bit2     e:4 --> bit5    */
    s1|= e&(0x7C000000L);   /* e:1-e:5 --> s:2-s:6              */

    e>>= 2;                 /* e:4 --> bit7     e:8 --> bit11   */
    s1|= e&(0x3F00000L);    /* e:4-e:9 --> s:7-s:12             */

    e>>= 2;                 /* e:8 --> bit13    e:12--> bit17   */
    s1|= e&(0xFC000L);      /* e:8-e:13 --> s:13-s:18           */

    e>>= 2;                 /* e:12 --> bit19                   */
    s1|= e&(0x3F00L);       /* e:12-e:17 --> s:19-s:24          */

    e= pe; s2= 0;           /* reinitialisation                 */
    if (e&B01)  s2=B24;     /* s:24= e:1                        */

    e<<= 9;                 /* e:32 --> bit23   e:29 --> bit20  */
    s2|= e&(0x3E00L);       /* e:28-e:32 --> s:19-s:23          */

    e<<= 2;                 /* e:29 --> bit18   e:25--> bit14   */
    s2|= e&(0xFC000L);      /* e:24-e:29 --> s:13-s:18          */

    e<<= 2;                 /* e:25 --> bit12   e:21 --> bit8   */
    s2|= e&(0x3F00000L);    /* e:20-e:25 --> s:7-s:12           */

    e<<= 2;                 /* e:21 --> bit6                    */
    s2|= e&(0xFC000000L);   /* e:16-e:20 --> s:1-s:6            */

    *ps1= s1; *ps2= s2;
}


/*-----------------------------------------------------------*/
/*          substitution64()                                 */
/*-----------------------------------------------------------*/
/*  Subtitutions de 8 blocs de 6 bits en 8 blocs de 4bits
/*
/*  Entree  :   e1e2    (48 bits, 1 a 24, 25 a 48)
/*  Sortie  :   s       (32 bits)
/*
/*  pour i=1 a 8 (1 a 4 pour e1, 5 a 8 pour e2) :
/*  isoler le bloc b6 de 6 bits de e1 ou e2 (decalage),
/*  recuperer le bloc de 4 bits b4 dans la table si[b6],
/*  inserer b4 dans s (decalage).
/**/
static void substitution64(unsigned int pe1,unsigned int pe2,unsigned int *ps)
{
    register unsigned int     e, s;

    e= pe1;                      /* initialisation e1                             */
    e>>=8;                       /* supprimer les 8 bits de poids faible (bloc 1) */
    s= s1[(e&0xFC0000L)>>18];    /* isoler les 6 bits de poids faible, substituer
                                    par le bloc de 4 bits de s1, le mettre dans s */
    s<<=4;                       /* bloc de 6 bits suivant (bloc 2)               */
    s|= (s2[(e&0x3F000L)>>12]);  /* isoler les 6 bits de poids faible, substituer
                                    par le bloc de 4 bits de s2, le mettre dans s */
    s<<=4;                       /* bloc de 6 bits suivant (bloc 3)               */
    s|= (s3[(e&0xFC0)>>6]);      /* isoler les 6 bits de poids faible, substituer
                                    par le bloc de 4 bits de s3, le mettre dans s */
    s<<=4;                       /* bloc de 6 bits suivant (bloc 4)               */
    s|= (s4[e&0x3F]);            /* isoler les 6 bits de poids faible, substituer
                                    par le bloc de 4 bits de s4, le mettre dans s */
    e= pe2;                      /* initialisation e2                             */
    s<<=4; e>>=8;                /* supprimer les 8 bits de poids faible (bloc 5) */
    s|= (s5[(e&0xFC0000L)>>18]); /* isoler les 6 bits de poids faible, substituer
                                    par le bloc de 4 bits de s5, le mettre dans s */
    s<<=4;                       /* bloc de 6 bits suivant (bloc 6)               */
    s|= (s6[(e&0x3F000L)>>12]);  /* isoler les 6 bits de poids faible, substituer
                                    par le bloc de 4 bits de s6, le mettre dans s */
    s<<=4;                       /* bloc de 6 bits suivant (bloc 7)               */
    s|= (s7[(e&0xFC0)>>6]);      /* isoler les 6 bits de poids faible, substituer
                                    par le bloc de 4 bits de s7, le mettre dans s */
    s<<=4;                       /* bloc de 6 bits suivant (bloc 8)               */
    s|= (s8[e&0x3F]);            /* isoler les 6 bits de poids faible, substituer
                                    par le bloc de 4 bits de s8, le mettre dans s */
    *ps= s;
}

/*-----------------------------------------------------------*/
/*          substitution32()                                 */
/*-----------------------------------------------------------*/
/*  Subtitutions de 8 blocs de 6 bits en 8 blocs de 4bits
/*
/*  Entree  :   e1e2    (48 bits, 1 a 24, 25 a 48)
/*  Sortie  :   s   (32 bits)
/*
/*  pour i=1 a 8 (1 a 4 pour e1, 5 a 8 pour e2) :
/*  isoler le bloc b6 de 6 bits de e1 ou e2 (decalage),
/*  recuperer le bloc de 4 bits b4 dans la table si[b6],
/*  inserer b4 dans s (decalage).
/**/
static void substitution32(unsigned long pe1,unsigned long pe2,unsigned long *ps)
{
    register unsigned long    e, s;

    e= pe1;                      /* initialisation e1                             */
    e>>=8;                       /* supprimer les 8 bits de poids faible (bloc 1) */
    s= s1[(e&0xFC0000L)>>18];    /* isoler les 6 bits de poids faible, substituer
                                    par le bloc de 4 bits de s1, le mettre dans s */
    s<<=4;                       /* bloc de 6 bits suivant (bloc 2)               */
    s|= (s2[(e&0x3F000L)>>12]);  /* isoler les 6 bits de poids faible, substituer
                                    par le bloc de 4 bits de s2, le mettre dans s */
    s<<=4;                       /* bloc de 6 bits suivant (bloc 3)               */
    s|= (s3[(e&0xFC0)>>6]);      /* isoler les 6 bits de poids faible, substituer
                                    par le bloc de 4 bits de s3, le mettre dans s */
    s<<=4;                       /* bloc de 6 bits suivant (bloc 4)               */
    s|= (s4[e&0x3F]);            /* isoler les 6 bits de poids faible, substituer
                                    par le bloc de 4 bits de s4, le mettre dans s */
    e= pe2;                      /* initialisation e2                             */
    s<<=4; e>>=8;                /* supprimer les 8 bits de poids faible (bloc 5) */
    s|= (s5[(e&0xFC0000L)>>18]); /* isoler les 6 bits de poids faible, substituer
                                    par le bloc de 4 bits de s5, le mettre dans s */
    s<<=4;                       /* bloc de 6 bits suivant (bloc 6)               */
    s|= (s6[(e&0x3F000L)>>12]);  /* isoler les 6 bits de poids faible, substituer
                                    par le bloc de 4 bits de s6, le mettre dans s */
    s<<=4;                       /* bloc de 6 bits suivant (bloc 7)               */
    s|= (s7[(e&0xFC0)>>6]);      /* isoler les 6 bits de poids faible, substituer
                                    par le bloc de 4 bits de s7, le mettre dans s */
    s<<=4;                       /* bloc de 6 bits suivant (bloc 8)               */
    s|= (s8[e&0x3F]);            /* isoler les 6 bits de poids faible, substituer
                                    par le bloc de 4 bits de s8, le mettre dans s */
    *ps= s;
}


/*-----------------------------------------------------------*/
/*          permutation64()                                  */
/*-----------------------------------------------------------*/
/*  Permutation bloc de 32 bits
/*
/*  Entree  :   e   (32 bits)
/*  Sortie  :   e   (32 bits)
/*
/*  t1...t32 --> t16...t25
/*  s:1  =  e:16     <==>  si e:14=1   alors s:1=1
/*      ...........
/*  s:32 =  e:25     <==>  si e:25=1    alors s:32=1
/**/
static void permutation64(unsigned int *pe)
{
   register unsigned int    e, s=0;
   
   e= *pe;
   
   if (e&B16)  s|=B01;   if (e&B07)  s|=B02;
   if (e&B20)  s|=B03;   if (e&B21)  s|=B04;
   if (e&B29)  s|=B05;   if (e&B12)  s|=B06;
   if (e&B28)  s|=B07;   if (e&B17)  s|=B08;
   if (e&B01)  s|=B09;   if (e&B15)  s|=B10;
   if (e&B23)  s|=B11;   if (e&B26)  s|=B12;
   if (e&B05)  s|=B13;   if (e&B18)  s|=B14;
   if (e&B31)  s|=B15;   if (e&B10)  s|=B16;
   if (e&B02)  s|=B17;   if (e&B08)  s|=B18;
   if (e&B24)  s|=B19;   if (e&B14)  s|=B20;
   if (e&B32)  s|=B21;   if (e&B27)  s|=B22;
   if (e&B03)  s|=B23;   if (e&B09)  s|=B24;
   if (e&B19)  s|=B25;   if (e&B13)  s|=B26;
   if (e&B30)  s|=B27;   if (e&B06)  s|=B28;
   if (e&B22)  s|=B29;   if (e&B11)  s|=B30;
   if (e&B04)  s|=B31;   if (e&B25)  s|=B32;
   
   *pe= s;
}


/*-----------------------------------------------------------*/
/*          permutation32()                                  */
/*-----------------------------------------------------------*/
/*  Permutation bloc de 32 bits
/*
/*  Entree  :   e   (32 bits)
/*  Sortie  :   e   (32 bits)
/*
/*  t1...t32 --> t16...t25
/*  s:1  =  e:16     <==>  si e:14=1   alors s:1=1
/*      ...........
/*  s:32 =  e:25     <==>  si e:25=1    alors s:32=1
/**/
static void permutation32(unsigned long *pe)
{
   register unsigned long   e, s=0;
   
   e= *pe;
   
   if (e&B16)  s|=B01;   if (e&B07)  s|=B02;
   if (e&B20)  s|=B03;   if (e&B21)  s|=B04;
   if (e&B29)  s|=B05;   if (e&B12)  s|=B06;
   if (e&B28)  s|=B07;   if (e&B17)  s|=B08;
   if (e&B01)  s|=B09;   if (e&B15)  s|=B10;
   if (e&B23)  s|=B11;   if (e&B26)  s|=B12;
   if (e&B05)  s|=B13;   if (e&B18)  s|=B14;
   if (e&B31)  s|=B15;   if (e&B10)  s|=B16;
   if (e&B02)  s|=B17;   if (e&B08)  s|=B18;
   if (e&B24)  s|=B19;   if (e&B14)  s|=B20;
   if (e&B32)  s|=B21;   if (e&B27)  s|=B22;
   if (e&B03)  s|=B23;   if (e&B09)  s|=B24;
   if (e&B19)  s|=B25;   if (e&B13)  s|=B26;
   if (e&B30)  s|=B27;   if (e&B06)  s|=B28;
   if (e&B22)  s|=B29;   if (e&B11)  s|=B30;
   if (e&B04)  s|=B31;   if (e&B25)  s|=B32;
   
   *pe= s;
}

/*-----------------------------------------------------------*/
/*          finalperm64()                                    */
/*-----------------------------------------------------------*/
/*  Permutation Finale
/*
/*  Entree  :   e1e2 (64 bits, 1 a 32, 33 a 64)
/*  Sortie  :   s1s2 (64 bits, 1 a 32, 33 a 64)
/*
/*  t1t2...t64 --> t40t8...t27
/*  s:1 (s1:1)    =  e:40 (e2:8)     <==>  si e2:8=1   alors s1:1=1
/*  s:2 (s1:2)    =  e:8  (e1:8)     <==>  si e1:8=1   alors s1:2=1
/*                ...........
/*  s:64 (s2:32)  =  e:27 (e1:27)    <==>  si e2:27=1  alors s2:32=1
/**/
static void finalperm64(unsigned int pe1,unsigned int pe2,unsigned int *ps1,unsigned int *ps2)
{
   register unsigned int    e1, e2, s1=0, s2=0;
   
   e1= pe1; e2= pe2;
   
   if (e2&B08)  s1|=B01;    if (e2&B16)  s1|=B03;
   if (e2&B24)  s1|=B05;    if (e2&B32)  s1|=B07;
   if (e2&B07)  s1|=B09;    if (e2&B15)  s1|=B11;
   if (e2&B23)  s1|=B13;    if (e2&B31)  s1|=B15;
   if (e2&B06)  s1|=B17;    if (e2&B14)  s1|=B19;
   if (e2&B22)  s1|=B21;    if (e2&B30)  s1|=B23;
   if (e2&B05)  s1|=B25;    if (e2&B13)  s1|=B27;
   if (e2&B21)  s1|=B29;    if (e2&B29)  s1|=B31;
   
   if (e2&B04)  s2|=B01;    if (e2&B12)  s2|=B03;
   if (e2&B20)  s2|=B05;    if (e2&B28)  s2|=B07;
   if (e2&B03)  s2|=B09;    if (e2&B11)  s2|=B11;
   if (e2&B19)  s2|=B13;    if (e2&B27)  s2|=B15;
   if (e2&B02)  s2|=B17;    if (e2&B10)  s2|=B19;
   if (e2&B18)  s2|=B21;    if (e2&B26)  s2|=B23;
   if (e2&B01)  s2|=B25;    if (e2&B09)  s2|=B27;
   if (e2&B17)  s2|=B29;    if (e2&B25)  s2|=B31;
   
   if (e1&B04)  s2|=B02;    if (e1&B12)  s2|=B04;
   if (e1&B20)  s2|=B06;    if (e1&B28)  s2|=B08;
   if (e1&B03)  s2|=B10;    if (e1&B11)  s2|=B12;
   if (e1&B19)  s2|=B14;    if (e1&B27)  s2|=B16;
   if (e1&B02)  s2|=B18;    if (e1&B10)  s2|=B20;
   if (e1&B18)  s2|=B22;    if (e1&B26)  s2|=B24;
   if (e1&B01)  s2|=B26;    if (e1&B09)  s2|=B28;
   if (e1&B17)  s2|=B30;    if (e1&B25)  s2|=B32;
   
   if (e1&B08)  s1|=B02;    if (e1&B16)  s1|=B04;
   if (e1&B24)  s1|=B06;    if (e1&B32)  s1|=B08;
   if (e1&B07)  s1|=B10;    if (e1&B15)  s1|=B12;
   if (e1&B23)  s1|=B14;    if (e1&B31)  s1|=B16;
   if (e1&B06)  s1|=B18;    if (e1&B14)  s1|=B20;
   if (e1&B22)  s1|=B22;    if (e1&B30)  s1|=B24;
   if (e1&B05)  s1|=B26;    if (e1&B13)  s1|=B28;
   if (e1&B21)  s1|=B30;    if (e1&B29)  s1|=B32;
   
   *ps1= s1; *ps2= s2;
}

/*-----------------------------------------------------------*/
/*          finalperm32()                                    */
/*-----------------------------------------------------------*/
/*  Permutation Finale
/*
/*  Entree  :   e1e2 (64 bits, 1 a 32, 33 a 64)
/*  Sortie  :   s1s2 (64 bits, 1 a 32, 33 a 64)
/*
/*  t1t2...t64 --> t40t8...t27
/*  s:1 (s1:1)    =  e:40 (e2:8)     <==>  si e2:8=1   alors s1:1=1
/*  s:2 (s1:2)    =  e:8  (e1:8)     <==>  si e1:8=1   alors s1:2=1
/*                ...........
/*  s:64 (s2:32)  =  e:27 (e1:27)    <==>  si e2:27=1  alors s2:32=1
/**/
static void finalperm32(unsigned long pe1,unsigned long pe2,unsigned long *ps1,unsigned long *ps2)
{
   register unsigned long   e1, e2, s1=0, s2=0;
   
   e1= pe1; e2= pe2;
   
   if (e2&B08)  s1|=B01;    if (e2&B16)  s1|=B03;
   if (e2&B24)  s1|=B05;    if (e2&B32)  s1|=B07;
   if (e2&B07)  s1|=B09;    if (e2&B15)  s1|=B11;
   if (e2&B23)  s1|=B13;    if (e2&B31)  s1|=B15;
   if (e2&B06)  s1|=B17;    if (e2&B14)  s1|=B19;
   if (e2&B22)  s1|=B21;    if (e2&B30)  s1|=B23;
   if (e2&B05)  s1|=B25;    if (e2&B13)  s1|=B27;
   if (e2&B21)  s1|=B29;    if (e2&B29)  s1|=B31;
   
   if (e2&B04)  s2|=B01;    if (e2&B12)  s2|=B03;
   if (e2&B20)  s2|=B05;    if (e2&B28)  s2|=B07;
   if (e2&B03)  s2|=B09;    if (e2&B11)  s2|=B11;
   if (e2&B19)  s2|=B13;    if (e2&B27)  s2|=B15;
   if (e2&B02)  s2|=B17;    if (e2&B10)  s2|=B19;
   if (e2&B18)  s2|=B21;    if (e2&B26)  s2|=B23;
   if (e2&B01)  s2|=B25;    if (e2&B09)  s2|=B27;
   if (e2&B17)  s2|=B29;    if (e2&B25)  s2|=B31;
   
   if (e1&B04)  s2|=B02;    if (e1&B12)  s2|=B04;
   if (e1&B20)  s2|=B06;    if (e1&B28)  s2|=B08;
   if (e1&B03)  s2|=B10;    if (e1&B11)  s2|=B12;
   if (e1&B19)  s2|=B14;    if (e1&B27)  s2|=B16;
   if (e1&B02)  s2|=B18;    if (e1&B10)  s2|=B20;
   if (e1&B18)  s2|=B22;    if (e1&B26)  s2|=B24;
   if (e1&B01)  s2|=B26;    if (e1&B09)  s2|=B28;
   if (e1&B17)  s2|=B30;    if (e1&B25)  s2|=B32;
   
   if (e1&B08)  s1|=B02;    if (e1&B16)  s1|=B04;
   if (e1&B24)  s1|=B06;    if (e1&B32)  s1|=B08;
   if (e1&B07)  s1|=B10;    if (e1&B15)  s1|=B12;
   if (e1&B23)  s1|=B14;    if (e1&B31)  s1|=B16;
   if (e1&B06)  s1|=B18;    if (e1&B14)  s1|=B20;
   if (e1&B22)  s1|=B22;    if (e1&B30)  s1|=B24;
   if (e1&B05)  s1|=B26;    if (e1&B13)  s1|=B28;
   if (e1&B21)  s1|=B30;    if (e1&B29)  s1|=B32;
   
   *ps1= s1; *ps2= s2;
}
